/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* Arnold JukeBox */
/* TODO: Testing! */
/* fbe1, fbe2, fbe3, fbe0 */

/*
fbe0 is used to read settings from board:

dip switches define bits 7..4.

off = 1, on = 0

then lower 4 bits are defined by a dial (0-15).

bit 7 is unused (appears to be locked to 0 on the pcb). it is in the on position?

bit 6,5 is timer value (software). it is the time between choosing next cart on the main screen.
(timer enabled must also be set)

off off = 8			1 1
off on = 16			1 0
on off = 24			0 1
on on = 32 			0 0

bit 4 = timer on/off

bit 3 ignored

bit 2..0: timer value when game manually selected (0-15)

0 = forever?
1 = 8s,
7 = 56s
(
*/

/* fbe2 is read */
/* bits 7..3 are current cartridge value, lower 4 bits are not used */

/* fbe1

bits 7,6: ?
bits 5..0 are cartridge number
*/

/* fbe3 is write */
/* bits 7..4 are cartridge */
/* bits 0..3: 3 bit value, or 2 bit value */

/* an array of cartridges */
/* first element is actually the jukebox cartridge itself */
#include "cpc.h"
#include "csd.h"
#include "asic.h"
#include "emudevice.h"

typedef struct
{
	unsigned char *pFilename;
	unsigned char *pData;
	unsigned long nLength;
	BOOL bEnabled;
} Jukebox_Cartridge;

typedef struct
{
	int CurrentCartridge;

	int SelectedCartridge;

	BOOL TimerActive;
	int CurrentTimerValue;

	/* no timer */
	/* time is set to longest */
	/* manual time of max is set */
	/* auto time of max is set */
	unsigned char SettingsInput;

	BOOL EnableState;


	/* 12 cartridges */
	Jukebox_Cartridge Cartridges[14];
} Jukebox;

static Jukebox box;

void Jukebox_SetTimerSelection( int nValue )
{
	box.SettingsInput &= ~( 0x03 << 5 );
	box.SettingsInput |= ( ( nValue ^ 0xff ) & 0x03 ) << 5;
}

int Jukebox_GetTimerSelection()
{
	return ( ( ( box.SettingsInput >> 5 ) ^ 0x0ff ) & 0x03 );

}

void Jukebox_SetManualTimerSelection( int nValue )
{
	box.SettingsInput &= ~0x0f;
	box.SettingsInput |= ( ( nValue ^ 0x0f ) & 0x0f );
}


int Jukebox_GetManualTimerSelection()
{
	return ( box.SettingsInput ^ 0x0f ) & 0x0f;
}

void Jukebox_SetTimerEnabled( BOOL bEnabled )
{
	if ( bEnabled )
	{
		box.SettingsInput &= ~( 1 << 4 );
	}
	else
	{
		box.SettingsInput |= ( 1 << 4 );
	}
}

BOOL Jukebox_IsTimerEnabled()
{
	return ( ( box.SettingsInput & ( 1 << 4 ) ) == 0 );

}

int Jukebox_GetSettingsInput()
{
	return box.SettingsInput;
}


static CPCPortRead JukeboxRead[1] =
{
	{
		0x0ffe0,
		0x0fbe0,
		Jukebox_Read
	}
};


static CPCPortWrite JukeboxWrite[1] =
{
	{
		0x0ffe0,
		0x0fbe0,
		Jukebox_Write
	}

};


void Jukebox_Update( void )
{
	if ( box.TimerActive )
	{
		box.CurrentTimerValue--;
		if ( box.CurrentTimerValue == 0 )
		{
			box.TimerActive = FALSE;

			Computer_RestartReset();
		}
	}
}


BOOL Jukebox_IsEnabled()
{
	return box.EnableState;
}


void Jukebox_Power( void )
{
	box.SelectedCartridge = 0;

	Jukebox_SelectCartridge( 0 );
	box.TimerActive = FALSE;
}

void Jukebox_Reset( void )
{
	Jukebox_SelectCartridge( 0 );
	box.TimerActive = FALSE;

}

void Jukebox_SetCartridgeEnable( int nCartridge, BOOL bEnable )
{
	if ( nCartridge <= 12 )
	{
		box.Cartridges[ nCartridge ].bEnabled = bEnable;
	}
}

BOOL Jukebox_IsCartridgeEnabled( int nCartridge )
{
	return box.Cartridges[ nCartridge ].bEnabled;
}

/* up to 12 cartridges */
void JukeboxDevice_Init(void)
{
	int i;

	for ( i = 0; i < 14; i++ )
	{
		box.Cartridges[ i ].pData = NULL;
		box.Cartridges[ i ].nLength = 0;
	}

	box.Cartridges[ 0 ].bEnabled = TRUE;
	box.EnableState = FALSE;

	box.CurrentCartridge = 0;
	box.SelectedCartridge = 0;

	box.TimerActive = FALSE;
	box.CurrentTimerValue = 0;

	box.EnableState = FALSE;

	/* no timer */
	/* time is set to longest */
	/* manual time of max is set */
	/* auto time of max is set */
	box.SettingsInput = (1 << 4);
}

void Jukebox_CartridgeInsert( int nCartridge, const unsigned char *pData, const unsigned long nLength )
{
	box.Cartridges[ nCartridge ].pData = (unsigned char *)malloc( nLength );
	if (box.Cartridges[ nCartridge ].pData != NULL )
	{
		memcpy(box.Cartridges[ nCartridge ].pData, pData, nLength );
	}
	box.Cartridges[ nCartridge ].nLength = nLength;
}

void Jukebox_CartridgeRemove( int nCartridge )
{
	if (box.Cartridges[ nCartridge ].pData != NULL )
	{
		free(box.Cartridges[ nCartridge ].pData );
		box.Cartridges[ nCartridge ].pData = NULL;
	}
	box.Cartridges[ nCartridge ].nLength = 0;

	Computer_RethinkMemory();
}

void Jukebox_InsertSystemCartridge( const unsigned char *pCartridgeData, const unsigned long CartridgeLength )
{
	Jukebox_CartridgeInsert( 0, pCartridgeData, CartridgeLength );

}

void JukeboxDevice_Finish(void)
{
	int i;

	for ( i = 0; i <= 12; i++ )
	{

		if ( box.Cartridges[ i ].pData != NULL )
		{
			free(box.Cartridges[ i ].pData );
			box.Cartridges[ i ].pData = NULL;
		}
		box.Cartridges[ i ].nLength = 0;
	}
}

void Jukebox_MemoryRethink( MemoryData *pData )
{
	pData; /* not used */

	if (
		/* cartridge index is valid? */
		( box.CurrentCartridge > 12 ) ||
		/* cartridge data loaded? */
		(box.Cartridges[ box.CurrentCartridge ].pData == NULL ) ||
		/* cartridge enabled in UI? */
		( !box.Cartridges[ box.CurrentCartridge ].bEnabled )
		)
	{
		/* if no cartridge exists here then map empty memory */
		Cartridge_RemoveI();
	}
	else
	{
		/* insert the cartridge */

		/* CPR? */
		if ( Cartridge_ValidateCartridge(box.Cartridges[ box.CurrentCartridge ].pData, box.Cartridges[ box.CurrentCartridge ].nLength ) )
		{
			/* insert it */
			Cartridge_Insert(box.Cartridges[ box.CurrentCartridge ].pData, box.Cartridges[ box.CurrentCartridge ].nLength );
		}
		else
		{
			/* insert as binary */
			Cartridge_InsertBinary(box.Cartridges[ box.CurrentCartridge ].pData, box.Cartridges[ box.CurrentCartridge ].nLength );
		}
	}

}

void Jukebox_SelectCartridge( int nCartridge )
{

	/* 0x0c0 what does that mean? */

	box.CurrentCartridge = nCartridge & 0x01f;

	Computer_RethinkMemory();

}

void Jukebox_Write( Z80_WORD nAddr, Z80_BYTE nValue )
{
	switch ( nAddr & 0x03 )
	{
		/* don't know if 0 or 2 can be written; CSD system cartridge doesn't read them */

		default
			:
	case 0:
	case 2:
		break;

	case 3:
	{
		/* store selected cartridge */
		box.SelectedCartridge = ( nValue >> 4 ) & 0x0f;
		/* set timer value */
		box.CurrentTimerValue = ( nValue & 0x0f ) * 50 * 8;
	}
	break;

	case 1:
	{
		/* assume 0x0c0 means start timer, not sure which bit */
		if ( ( nValue & 0x0c0 ) != 0 )
		{
			box.TimerActive = TRUE;
		}
		else
		{
			box.TimerActive = FALSE;
		}

		box.CurrentCartridge = nValue & 0x01f;
		Jukebox_SelectCartridge( box.CurrentCartridge );
	}
	break;
	}
}


BOOL Jukebox_Read( Z80_WORD nAddr, Z80_BYTE *pDeviceData )
{
	switch ( nAddr & 0x03 )
	{
	case 0:
		*pDeviceData = box.SettingsInput;
		return TRUE;

	case 2:
		*pDeviceData = ( box.SelectedCartridge << 4 );
		return TRUE;

	default:
		break;
	}
	return FALSE;
}


static EmuDevice JukeboxDevice =
{
	NULL,
	JukeboxDevice_Init,
	JukeboxDevice_Finish,
	"Jukebox",
	"Jukebox",
	"Amstrad CSD",
	CONNECTION_EXPANSION,   /* connects to expansion */
	0,	
	1,
	JukeboxRead,					/* no read ports */
	1,
	JukeboxWrite,
	0,                /* no memory read*/
	NULL,
	0,                /* no memory write */
	NULL,
	Jukebox_Reset,
	Jukebox_MemoryRethink,
	Jukebox_Reset,
	0,
	NULL,
	0,                      /* 2 buttons */
	NULL,
	0,                      /* 1 onboard roms */
	NULL,
	0,
	NULL,
	NULL,                   /* no cursor function */
	NULL,                   /* no generic roms */
	NULL,					/* printer */
	NULL,					/* joystick */
	0,
	NULL,					/* memory ranges */
	NULL, /* sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	NULL, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL,
};

int Jukebox_Init()
{
	return RegisterDevice(&JukeboxDevice);
}

